﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using Microsoft.Xna.Framework;

namespace GE.World.Entities
{
    class BulletEntity : WorldEntity
    {
        /// <summary>
        /// Bullet's texture
        /// </summary>
        int _iIdTexture;

        /// <summary>
        /// Bullet's sprite
        /// </summary>
        int _iIdSprite;

        /// <summary>
        /// Shape for collision detection
        /// </summary>
        DynamicShapeRectangle _shape;

        /// <summary>
        /// Bullet's speed
        /// </summary>
        Vector2 _v2Speed;

        /// <summary>
        /// Bullet's life time in milliseconds
        /// </summary>
        int _iLifeTime;

        /// <summary>
        /// Bullet's birth time
        /// </summary>
        int _iBirthTime;

        /// <summary>
        /// Constructor
        /// </summary>
        public BulletEntity()
            : base()
        {
            _iIdTexture = -1;
            _iIdSprite = -1;

            _shape = Physics.Physics.Instance.createDynamicRectangle(0, 0, Vector2.Zero, this);
            _shape._bCollisionEnable = false;

            _v2Speed = Vector2.Zero;
            _v2Position = Vector2.Zero;
            _iLifeTime = 0;
            _iBirthTime = 0;
            _iDamages = 0;
        }

        /// <summary>
        /// Activator
        /// </summary>
        /// <param name="idTexture"></param>
        /// <param name="idSprite"></param>
        /// <param name="lifeTime"></param>
        /// <param name="damages"></param>
        /// <param name="speed"></param>
        /// <param name="position"></param>
        public void activate(int idTexture, int idSprite, int lifeTime, int damages, Vector2 speed, Vector2 position, eSide side)
        {
            _iIdTexture = idTexture;
            _iIdSprite = idSprite;
            _iLifeTime = lifeTime;
            _iDamages = damages;
            _v2Speed = speed;
            _v2Position = position;
            _iBirthTime = TimeClock.Clock.instance.millisecs;
            _side = side;
            _shape.resize(Visu.Instance.getSpriteWidth(_iIdTexture, _iIdSprite), Visu.Instance.getSpriteHeight(_iIdTexture, _iIdSprite));
            _shape._bCollisionEnable = true;
            if(side == eSide.eSidePlayer)
                _shape._iGroup = (int)ePhysicGroup.ePhysicBullet;
            else
                _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;
            _bActive = true;
        }

        /// <summary>
        /// Update
        /// </summary>
        public override void update()
        {
            //check the life time
            if (TimeClock.Clock.instance.millisecs > _iBirthTime + _iLifeTime) die();

            //move the bullet
            Position += _v2Speed;
            _shape._v2position = Position;

            //if the bullet is on the player side, check collision with enemies
            if (_side == eSide.eSidePlayer)
            {
                CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicEnemy);
                if (res == null) return;

               
                res.Entity.hurt(_iDamages);
                die();
            }
            else //enemy bullet
            {
                CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer);
                if (res == null) return;

                die();
                res.Entity.hurt(_iDamages);
            }
        }

        /// <summary>
        /// Render
        /// </summary>
        public override void render()
        {
            Visu.Instance.displaySprite(_iIdTexture, _iIdSprite, ScreenPosition);
        }


        /// <summary>
        /// Die
        /// </summary>
        public override void die()
        {
            _shape._bCollisionEnable = false;
            _bActive = false;
        }

        public override void hurt(int damages)
        {
            die();
        }
    }
}
